/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.

	modified to work with the G400  - Mark Watson
*/

#ifndef DRIVERINTERFACE_H
#define DRIVERINTERFACE_H

#include <Accelerant.h>
#include <Drivers.h>
#include <PCI.h>
#include <OS.h>

/*
	This is the info that needs to be shared between the kernel driver and
	the accelerant for the g400 driver.
*/
#if defined(__cplusplus)
extern "C" {
#endif

typedef struct {
	sem_id	sem;
	int32	ben;
} benaphore;

#define INIT_BEN(x)		x.sem = create_sem(0, "G400 "#x" benaphore");  x.ben = 0;
#define AQUIRE_BEN(x)	if((atomic_add(&(x.ben), 1)) >= 1) acquire_sem(x.sem);
#define RELEASE_BEN(x)	if((atomic_add(&(x.ben), -1)) > 1) release_sem(x.sem);
#define	DELETE_BEN(x)	delete_sem(x.sem);


#define G400_PRIVATE_DATA_MAGIC	0x1234 /* a private driver rev, of sorts */

#define MAX_G400_DEVICE_NAME_LENGTH 32

#define SKD_MOVE_CURSOR    0x00000001
#define SKD_PROGRAM_CLUT   0x00000002
#define SKD_SET_START_ADDR 0x00000004
#define SKD_SET_CURSOR     0x00000008
#define SKD_HANDLER_INSTALLED 0x80000000

enum {
	G400_GET_PRIVATE_DATA = B_DEVICE_OP_CODES_END + 1,
	G400_GET_PCI,
	G400_SET_PCI,
	G400_DEVICE_NAME,
	G400_RUN_INTERRUPTS
};

/*shared info*/
typedef struct {
  /*a few ID things*/
	uint16	vendor_id;	/* PCI vendor ID, from pci_info */
	uint16	device_id;	/* PCI device ID, from pci_info */
	uint8	revision;	/* PCI device revsion, from pci_info */

  /* big workaround for 4.5.0 */
	uint32 use_clone_bugfix;	/*for 4.5.0*/
	uint32 * clone_bugfix_regs;

  /*memory mappings*/
	area_id	regs_area;	/* Kernel's area_id for the memory mapped registers.
							It will be cloned into the accelerant's	address
							space. */

	area_id	fb_area;	/* Frame buffer's area_id.  The addresses are shared
							with all teams. */
	void	*framebuffer;	/* As viewed from virtual memory */
	void	*framebuffer_pci;	/* As viewed from the PCI bus (for DMA) */

  /*screenmodees*/
	area_id	mode_area;	/* Contains the list of display modes the driver supports */
	uint32	mode_count;	/* Number of display modes in the list */

  /*flags - used by driver*/
	uint32 flags;

  /*vblank semaphore*/
	sem_id	vblank;	/* The vertical blank semaphore.  Ownership will be
						transfered to the team opening the device first */
  /*cursor information*/
	struct {
		uint8*	data;		/*  Pointer into the frame buffer to where the
								cursor data starts */
		uint16	hot_x;		/* Cursor hot spot. The top left corner of the cursor */
		uint16	hot_y;		/* is 0,0 */
		uint16	x;			/* The location of the cursor hot spot on the */
		uint16	y;			/* display (or desktop?) */
		uint16	width;		/* Width and height of the cursor shape */
		uint16	height;
		bool	is_visible;	/* Is the cursor currently displayed? */
	}		cursor;

  /*Vblank times*/
	bigtime_t	refresh_period;	/* Duration of one frame (ie 1/refresh rate) */
	bigtime_t	blank_period;	/* Duration of the blanking period.   These are
									usefull when faking vertical blanking*/

  /*colour lookup table*/
	uint8	color_data[3 * 256];	/* Colour lookup table - as used by DAC */

  /*more display mode stuff*/
	display_mode dm;		/* current display mode configuration */

  /*frame buffer config - for BDirectScreen*/
	frame_buffer_config fbc;	/* bytes_per_row and start of frame buffer */
  /*acceleration engine*/
	struct {
		uint32		count;		/* last dwgsync slot used */
		uint32		last_idle;	/* last dwgsync slot we *know* the engine was idle after */ 
		benaphore	lock;		/* for serializing access to the acceleration engine */
	}		engine;

  /*mirror of the ROM*/
	uint8 rom_mirror[32768];

  /*PINS info - information gathered from PINS (and other sources ATM)*/
	struct
	{
		/*specialised registers for initialisation*/
		uint32 mem_ctl;
		uint32 mem_type;
		uint32 mem_rd;
		uint32 mem_rfhcnt;

		/*general card information*/
		uint32 card_type;                      /*1=g400,2=g400MAX*/
		uint32 pix_clk_max;	               /*max pix clk speed (Hz) - same for all bit depths*/
		uint32 mem_clk_period;                 /*memory clock period (ns)*/
		uint32 mem_size;                       /*Frame buffer memory, in bytes. */
		uint8 sdram;                           /*1 if SGRAM card */

		/*system clocks*/
		uint8 syspll_m;
		uint8 syspll_n;
		uint8 syspll_mclk_div;
		uint8 syspll_oclk_div;
	} ps;
} shared_info;

/* Read or write a value in PCI configuration space */
typedef struct {
	uint32	magic;		/* magic number to make sure the caller groks us */
	uint32	offset;		/* Offset to read/write */
	uint32	size;		/* Number of bytes to transfer */
	uint32	value;		/* The value read or written */
} g400_get_set_pci;

/* Set some boolean condition (like enabling or disabling interrupts) */
typedef struct {
	uint32	magic;		/* magic number to make sure the caller groks us */
	bool	do_it;		/* state to set */
} g400_set_bool_state;

/* Retrieve the area_id of the kernel/accelerant shared info */
typedef struct {
	uint32	magic;		/* magic number to make sure the caller groks us */
	area_id	shared_info_area;	/* area_id containing the shared information */
} g400_get_private_data;

/* Retrieve the device name.  Usefull for when we have a file handle, but want
to know the device name (like when we are cloning the accelerant) */
typedef struct {
	uint32	magic;		/* magic number to make sure the caller groks us */
	char	*name;		/* The name of the device, less the /dev root */
} g400_device_name;

enum {
	G400_WAIT_FOR_VBLANK = (1 << 0)
};

#if defined(__cplusplus)
}
#endif


#endif
