/*program the DAC*/
/*Mark Watson 2/2000*/

#include "mga_std.h"

/*set the mode, brightness is a value from 0->2 (where 1 is equivalent to direct)*/
status_t g400_dac_mode(int mode,float brightness)
{
	uint8 *r,*g,*b,t[64];
	int i;

	/*set colour arrays to point to space reserved in shared info*/
	r=si->color_data;
	g=r+256;
	b=g+256;

	LOG(1,"DAC:Setting screen mode\n");
	/*init a basic palette for brightness specified*/
	for (i=0;i<256;i++)
	{
		r[i]=i*brightness;
		if (r[i]>255) r[i]=255;
	}

	/*modify the palette for the specified mode (&validate mode)*/
	switch(mode)
	{
	case BPP8:
	case BPP24:case BPP32:
		for (i=0;i<256;i++)
		{
			b[i]=g[i]=r[i];
		}
		break;
	case BPP16:
		for (i=0;i<64;i++)
		{
			t[i]=r[i<<2];
		}
		for (i=0;i<64;i++)
		{
			g[i]=t[i];
		}
		for (i=0;i<32;i++)
		{
			b[i]=r[i]=t[i<<1];
		}
		break;
	case BPP15:
		for (i=0;i<32;i++)
		{
			t[i]=r[i<<3];
		}
		for (i=0;i<32;i++)
		{
			g[i]=r[i]=b[i]=t[i];
		}
		break;
	case BPP32DIR:
		break;
	default:
		LOG(3,"DAC:Invalid bit depth requested\n");
		return B_ERROR;
		break;
	}

	if (g400_dac_palette(r,g,b)!=B_OK) return B_ERROR;

	/*set VCLK scaling - sort of CRTC, but fits better here*/
	switch(mode)
	{
	case BPP8:
		VGAW_I(CRTCEXT,3,0x80);
		break;
	case BPP15:case BPP16:
		VGAW_I(CRTCEXT,3,0x81);
		break;
	case BPP24:
		VGAW_I(CRTCEXT,3,0x82);
		break;
	case BPP32:case BPP32DIR:
		VGAW_I(CRTCEXT,3,0x83);
		break;
	default:
		LOG(3,"DAC:Invalid bit depth\n");
		return B_ERROR;
		break;
	}

	/*set the mode - also sets VCLK dividor*/
	DXIW(MULCTRL,mode);

	LOG(2,"DAC: mulctrl=%x\n",DXIR(MULCTRL));

	return B_OK;
}

/*program the DAC palette using the given r,g,b values*/
status_t g400_dac_palette(uint8 r[256],uint8 g[256],uint8 b[256])
{
	int i;

	LOG(1,"DAC: setting palette\n");
	/*clear palwtadd to start programming*/
	DACW(PALWTADD,0);

	/*loop through all 256 to program DAC*/
	for (i=0;i<256;i++)
	{
		DACW(PALDATA,r[i]);
		DACW(PALDATA,g[i]);
		DACW(PALDATA,b[i]);
	}

	if (DACR(PALWTADD)!=0)
	{
		LOG(3,"DAC: PALWTADD is not 0 after programming\n");
		return B_ERROR;
	}

	return B_OK;
}

status_t g400_dac_cursor_init()
{
	int i;
	uint32 * fb;

	/*store cursor at the start of the framebuffer*/
	DXIW(CURADDL,0);/*data at start of framebuffer*/
	DXIW(CURADDH,0);
	DXIW(CURCTRL,1);

	/*set cursor colour*/
	DXIW(CURCOL0RED,0XFF);
	DXIW(CURCOL0GREEN,0xFF);
	DXIW(CURCOL0BLUE,0xFF);
	DXIW(CURCOL1RED,0);
	DXIW(CURCOL1GREEN,0);
	DXIW(CURCOL1BLUE,0);
	DXIW(CURCOL2RED,0);
	DXIW(CURCOL2GREEN,0);
	DXIW(CURCOL2BLUE,0);

	/*clear cursor*/
	fb = (uint32 *) si->framebuffer;
	for (i=0;i<(1024/4);i++)
	{
		fb[i]=0;
	}
	return B_OK;
}

status_t g400_dac_cursor_show()
{
	DXIW(CURCTRL,1);
	return B_OK;
}

status_t g400_dac_cursor_hide()
{
	DXIW(CURCTRL,0);
	return B_OK;
}

/*set up cursor shape*/
status_t g400_dac_cursor_define(uint8* andMask,uint8* xorMask)
{
	uint8 * cursor;
	int y;

	/*get a pointer to the cursor*/
	cursor = (uint8*) si->framebuffer;

	/*draw the cursor*/
	for(y=0;y<16;y++)
	{
		cursor[y*16+7]=~*andMask++;
		cursor[y*16+15]=*xorMask++;
		cursor[y*16+6]=~*andMask++;
		cursor[y*16+14]=*xorMask++;
	}

	return B_OK;
}

/*position the cursor*/
status_t g400_dac_cursor_position(uint16 x ,uint16 y)
{
	int i=64;
/*	LOG(1,"DAC: cursor-> %d %d\n",x,y); */

	x+=i;
	y+=i;

	DACW(CURSPOSXL,x&0xFF);
	DACW(CURSPOSXH,x>>8);
	DACW(CURSPOSYL,y&0xFF);
	DACW(CURSPOSYH,y>>8);

	return B_OK;
}

/*program the pixpll - frequency in kHz*/
/*important notes:
 * MISC(clksel) = select A,B,C PIXPLL (25,28,none)
 * PIXPLLC is used - others should be kept as is
 * VCLK is quadword clock (max is PIXPLL/2) - set according to DEPTH
 * BESCLK,CRTC2 are not touched 
 */
status_t g400_dac_set_pix_pll(float f_vco)
{
	uint8 m=0,n=0,p=0;
	uint8 seq1;

	float pix_setting;
	status_t result;

	LOG(1,"DAC:Setting PIX PLL\n");

	result = g400_dac_pix_pll_find(f_vco,&pix_setting,&m,&n,&p);
	if (result != B_OK)
	{
		return result;
	}
	
	/*turn off screen*/
	seq1=VGAR_I(SEQ,1);
	VGAW_I(SEQ,1,0x20);
	
	/*reprogram (disable,select,wait for stability,enable)*/
	DXIW(PIXCLKCTRL,DXIR(PIXCLKCTRL)|0x04);         /*disable the PIXPLL*/
	DXIW(PIXCLKCTRL,(DXIR(PIXCLKCTRL)&0x0C)|0x01);  /*select the PIXPLL*/
	VGAW(MISCW,((VGAR(MISCR)&0xF3)|0x8));           /*select PIXPLLC*/
	DXIW(PIXPLLCM,(m));				/*set m value*/
	DXIW(PIXPLLCN,(n));				/*set n value*/
	DXIW(PIXPLLCP,(p));                             /*set p value*/
	while(!(DXIR(PIXPLLSTAT)&0x40));                /*wait for the PIXPLL frequency to lock*/
	LOG(2,"DAC: PIX PLL frequency locked\n");
	DXIW(PIXCLKCTRL,DXIR(PIXCLKCTRL)&0x0B);         /*enable the PIXPLL*/

	/*restore screens to previous state*/
	VGAW_I(SEQ,1,seq1);

	return B_OK;
}

/*find nearest valid pix pll*/
status_t g400_dac_pix_pll_find(float f_vco,float * result,uint8 * m_result,uint8 * n_result,uint8 * p_result)
{
	float f_ref=27.000;
	int n_min=8;
	int n_max=128;
	int m_min=2;
	int m_max=32;
	int m=0,n=0,p=0;
	float error;
	float error_best;
	int best[3]; 
	float f_rat;

	LOG(1,"DAC:Checking PIX PLL\n");

	/*fail if too high*/
	if (f_vco>(si->ps.pix_clk_max/1000000.0))
	{
		LOG(3,"DAC: Attemped to set too high pixel clock\n");
		return B_ERROR;
	}

	/*f_rat.m/p=n where m=M+1,n=N+1,p=P+1,f_rat=f_vco/f_ref*/
	f_rat = f_vco/f_ref;
	error_best = 999999999;
	for (p=0x2 -1*(f_vco>80.0);p<0x10;p=p<<1)
	{
		for (m=m_min;m<m_max;m++)
		{
			/*calculate n for this m & p (and check for validity)*/
			n=(int)((f_rat*m*p)+0.5);
			if (n>n_max || n<n_min)
				continue;
		
			/*find error in frequency this gives*/
			error=fabs(((f_ref*n)/(m*p))-f_vco);
			if (error<error_best)
			{
				error_best = error;
				best[0]=m;
				best[1]=n;
				best[2]=p;
			}
		}
	}
	m=best[0];
	n=best[1];
	p=best[2];

	/*calculate value of s for fvco see 4.7.8.1*/
	for(;;)/*set loop filter bandwidth -> spot the PERL coder :-)*/
	{
		if(f_vco>240) {p|=0x18;break;};
		if(f_vco>170) {p|=0x10;break;};
		if(f_vco>110) {p|=0x08;break;};
		break;
	}

	/*set the result*/
	*result = (float) (f_ref*n)/(m*(p&0x7));
	*m_result = m-1;
	*n_result = n-1;
	*p_result = p-1;

	/*display the found value*/
	LOG(2,"PIXPLLCHECK: requested %fMHz got %fMHz\n",(double)f_vco,*result);

	return B_OK;
}


/*set up system pll - NB mclk is memory clock, */ 
status_t g400_dac_set_sys_pll(int m,int n,int mclk_div,int oclk_div)
{
	uint32 scalers; /*value for option 3*/
	uint8 p;        /*values for DAC sys pll registers*/

	uint8 mclk_duty,oclk_duty;
	int fvco;
	float mclk,oclk;
	float temp_f;

	float div[8]={0.3333,0.4,0.4444,0.5,0.6666,0,0,0};

	LOG(1,"DAC:Setting up system clock\n");

	/*calculate fvco from given input*/
	fvco = 27*(n+1)/(m+1);

	/*calculate value of p for fvco see 4.7.8.1*/
	p=0;
	for(;;)/*set loop filter bandwidth*/
	{
		if(fvco>240) {p|=0x18;break;};
		if(fvco>170) {p|=0x10;break;};
		if(fvco>110) {p|=0x08;break;};
		break;
	}

	/*calculate the real clock speed*/
	mclk = div[mclk_div]*(27*(n+1)/(m+1));
	oclk = div[oclk_div]*(27*(n+1)/(m+1));

	/*work out the duty cycle correction*/
	temp_f=1/(float)oclk;
	temp_f*=1000;
	LOG(2,"DAC:oclk correction ns: %f\n",temp_f);
	temp_f-=2.25;
	temp_f/=0.5;
	oclk_duty=(uint8)temp_f;

	temp_f=1/(float)mclk;
	temp_f*=1000;
	LOG(2,"DAC:mclk correction ns: %f\n",temp_f);
	temp_f-=2.25;
	temp_f/=0.5;
	mclk_duty=(uint8)temp_f;

	/*calculate OPTION3*/
	scalers=(0x1<<0)|(0x1<<10)|(0x1<<20);
	scalers|=(oclk_div<<3)|(mclk_div<<13)|(oclk_div<<23);
	scalers|=(oclk_duty<<6)|(mclk_duty<<16)|(oclk_duty<<26);

	/*print out the results*/
	LOG(2,"DAC: MCLK:%f\tOCLK:%f\n",mclk,oclk);
	LOG(2,"DAC: n:0x%x m:0x%x p:0x%x \n",n,m,p);
	LOG(2,"DAC: mclk_div:0x%x mclk_duty:0x%x\noclk_div:0x%x oclk_duty:0x%x\nOPTION3: 0x%x\n",mclk_div,mclk_duty,oclk_div,oclk_duty,scalers);

	/*reprogram the clock - set PCI/AGP, program, set to programmed*/
	CFGW(OPTION2,0);				/*clear so don't o/clock add ons*/
	CFGW(OPTION,CFGR(OPTION)|0x04);                 /*disable the SYSPLL*/
	CFGW(OPTION3,0);                                /*select the PCI/AGP clock*/
	CFGW(OPTION,CFGR(OPTION)&0xFFFFFFFB);           /*enable the SYSPLL*/

	DXIW(SYSPLLM,m);
	DXIW(SYSPLLN,n);
	DXIW(SYSPLLP,p);
	while(!(DXIR(SYSPLLSTAT)&0x40));                /*wait for the SYSPLL frequency to lock*/
	LOG(2,"DAC:System PLL locked\n");

	CFGW(OPTION,CFGR(OPTION)|0x04);                 /*disable the SYSPLL*/
	CFGW(OPTION3,scalers);                                /*select the SYSPLLs chosen*/
	CFGW(OPTION,CFGR(OPTION)&0xFFFFFFFB);           /*enable the SYSPLL*/

	return B_OK;
}

