/*CTRC functionality*/
/*Mark Watson 2/2000*/

#include "mga_std.h"

/*Adjust passed parameters to a valid mode line*/
status_t g400_crtc_validate_timing(
	uint16 *hd_e,uint16 *hs_s,uint16 *hs_e,uint16 *ht,
	uint16 *vd_e,uint16 *vs_s,uint16 *vs_e,uint16 *vt
)
{
/*horizontal*/
	/*make all parameters multiples of 8 and confine to required number of bits*/
	*hd_e&=0x7F8;
	*hs_s&=0xFF8;
	*hs_e&=0xFF8;
	*ht  &=0xFF8;

	/*confine to a reasonable width*/
	if (*hd_e<640) *hd_e=640;
	if (*hd_e>2048) *hd_e=2048;

	/*if horizontal total does not leave room for a sensible sync pulse, increase it!*/
	if (*ht<(*hd_e+80)) *ht=*hd_e+80;

	/*make sure sync pulse is not during display*/
	if (*hs_e>(*ht-0x8)) *hs_e=*ht-0x8;
	if (*hs_s<(*hd_e+0x8)) *hs_e=*hd_e+0x8;

	/*correct sync pulse if it is too long*/
	if (*hs_e>(*hs_s+0xF8)) *hs_e=*hs_s+0xF8;

	/*fail if they are now outside required number of bits*/
	if (
		*hd_e!=(*hd_e&0x7F8) ||
		*hs_s!=(*hs_s&0xFF8) ||
		*hs_e!=(*hs_e&0xFF8) ||
		*ht  !=(*ht  &0xFF8)
	)
	{
		LOG(3,"CRTC:Horizontal timing fell out of bits\n");
		return B_ERROR;
	}

/*vertical*/
	/*squish to required number of bits*/
	*vd_e&=0x7FF;
	*vs_s&=0xFFF;
	*vs_e&=0xFFF;
	*vt  &=0xFFF;

	/*confine to a reasonable height*/
	if (*vd_e<400) *vd_e=400;
	if (*vd_e>2048) *vd_e=2048;

	/*if vertical total does not leave room for a sync pulse, increase it!*/
	if (*vt<(*vd_e+3)) *vt=*vd_e+3;

	/*make sure sync pulse if not during display*/
	if (*vs_e>(*vt-1)) *vs_e=*vt-1;
	if (*vs_s<(*vd_e+1)) *vs_s=*vd_e+1;

	/*correct sync pulse if it is too long*/
	if (*vs_e>(*vs_s+0xF)) *vs_e=*vs_s+0xF;

	/*fail if now outside required number of bits*/
	if (
		*vd_e!=(*vd_e&0x7FF) ||
		*vs_s!=(*vs_s&0xFFF) ||
		*vs_e!=(*vs_e&0xFFF) ||
		*vt  !=(*vt  &0xFFF)
	)
	{
		LOG(3,"CRTC:Vertical timing fell out of bits\n");
		return B_ERROR;
	}

	return B_OK;
}
	

/*set a mode line - inputs are in pixels*/
status_t g400_crtc_set_timing(
	uint16 hd_e,uint16 hs_s,uint16 hs_e,uint16 ht,
	uint16 vd_e,uint16 vs_s,uint16 vs_e,uint16 vt,
	uint8 hsync_pos,uint8 vsync_pos
)
{
	uint32 htotal;		/*total horizontal total VCLKs*/
	uint32 hdisp_e;            /*end of horizontal display (begins at 0)*/
	uint32 hsync_s;            /*begin of horizontal sync pulse*/
	uint32 hsync_e;            /*end of horizontal sync pulse*/
	uint32 hblnk_s;            /*begin horizontal blanking*/
	uint32 hblnk_e;            /*end horizontal blanking*/

	uint32 vtotal;		/*total vertical total scanlines*/
	uint32 vdisp_e;            /*end of vertical display*/
	uint32 vsync_s;            /*begin of vertical sync pulse*/
	uint32 vsync_e;            /*end of vertical sync pulse*/
	uint32 vblnk_s;            /*begin vertical blanking*/
	uint32 vblnk_e;            /*end vertical blanking*/

	LOG(1,"CRTC: setting timing\n");

	/*Modify parameters as required by the G400*/
	htotal=(ht>>3)-5;
	hdisp_e=(hd_e>>3)-1;
	hsync_s=(hs_s>>3);
	hsync_e=(hs_e>>3);
	hblnk_s=hdisp_e;
	hblnk_e=htotal+4;
	
	vtotal=vt-2;
	vdisp_e=vd_e-1;
	vsync_s=vs_s-1;
	vsync_e=vs_e-1;
	vblnk_s=vdisp_e;
	vblnk_e=vtotal+1;

	/*log the mode I am setting*/
	LOG(2,"CRTC:\n\tHTOT:%x\n\tHDISPEND:%x\n\tHBLNKS:%x\n\tHBLNKE:%x\n\tHSYNCS:%x\n\tHSYNCE:%x\n\t",htotal,hdisp_e,hblnk_s,hblnk_e,hsync_s,hsync_e);
	LOG(2,"VTOT:%x\n\tVDISPEND:%x\n\tVBLNKS:%x\n\tVBLNKE:%x\n\tVSYNCS:%x\n\tVSYNCE:%x\n",vtotal,vdisp_e,vblnk_s,vblnk_e,vsync_s,vsync_e);

	/*actually program the card! Note linecomp is programmed to vlnk_s for VBI*/
	/*horizontal - VGA regs*/
	VGAW_I(CRTC,0,htotal&0xFF);
	VGAW_I(CRTC,1,hdisp_e&0xFF);
	VGAW_I(CRTC,2,hblnk_s&0xFF);
	VGAW_I(CRTC,3,(hblnk_e&0x1F)|0x80);
	VGAW_I(CRTC,4,hsync_s&0xFF);
	VGAW_I(CRTC,5,(hsync_e&0x1F)|((hblnk_e&0x20)<<2));

	/*vertical - VGA regs*/
	VGAW_I(CRTC,6,vtotal&0xFF);
	VGAW_I(CRTC,7,
	(
		((vtotal&0x100)>>8)|((vtotal&0x200)>>4)|
		((vdisp_e&0x100)>>7)|((vdisp_e&0x200)>>3)|
		((vsync_s&0x100)>>6)|((vsync_s&0x200)>>2)|
		((vblnk_s&0x100)>>5)
	));
	VGAW_I(CRTC,0x9,((vblnk_s&0x200)>>4));
	VGAW_I(CRTC,0x10,vsync_s&0xFF);
	VGAW_I(CRTC,0x11,((VGAR_I(CRTC,0x11))&0xF0)|(vsync_e&0xF));
	VGAW_I(CRTC,0x12,vdisp_e&0xFF);
	VGAW_I(CRTC,0x15,vblnk_s&0xFF);
	VGAW_I(CRTC,0x16,vblnk_e&0xFF);

	/*horizontal - extended regs*/
	VGAW_I(CRTCEXT,1,
	(
		((htotal&0x100)>>8)|
		((hblnk_s&0x100)>>7)|
		((hsync_s&0x100)>>6)|
		(hblnk_e&0x40)|
		(VGAR_I(CRTCEXT,1)&0x30)
	));

	/*vertical - extended regs*/
	VGAW_I(CRTCEXT,2,
	(
	 	((vtotal&0xC00)>>10)|
		((vdisp_e&0x400)>>8)|
		((vblnk_s&0xC00)>>7)|
		((vsync_s&0xC00)>>5)
	));

	/*set up HSYNC & VSYNC polarity*/
	VGAW(MISCW,(VGAR(MISCR)&0x3F)|((!vsync_pos)<<7)|((!hsync_pos)<<6));
	LOG(2,"HSYNC/VSYNC pol:%x %x MISC dump:%x\n",hsync_pos,vsync_pos,VGAR(MISCR));

	return B_OK;
}

status_t g400_crtc_dpms(uint8 display,uint8 h,uint8 v)
{
	VGAW_I(SEQ,1,(!display)<<5);
	VGAW_I(CRTCEXT,1,(VGAR_I(CRTCEXT,1)&0xCF)|((!v)<<5))|((!h)<<4);
	
	VGAW_I(CRTC,0x17,0xC3);/*do not force disable all syncs and other stuff*/
	VGAW_I(CRTC,0x14,0x00);

	return B_OK;
}

status_t g400_crtc_dpms_fetch(uint8 * display,uint8 * h,uint8 * v)
{
	*display=!((VGAR_I(SEQ,1)&0x20)>>5);
	*h=!((VGAR_I(CRTCEXT,1)&0x10)>>4);
	*v=!((VGAR_I(CRTCEXT,1)&0x20)>>5);

	return B_OK;
}

status_t g400_crtc_set_display_pitch(uint32 pitch,uint8 bpp) 
{
	uint32 offset;

	LOG(1,"CRTC: setting card pitch (offset between lines)\n");

	/*figure out offset value hardware needs*/
	offset = (pitch*bpp)/128;

	LOG(2,"CRTC: offset: %x\n",offset);

	/*program the card!*/
	VGAW_I(CRTC,0x13,(offset&0xFF));
	VGAW_I(CRTCEXT,0,(VGAR_I(CRTCEXT,0)&0xCF)|((offset&0x300)>>4));
	return B_OK;
}

status_t g400_crtc_set_display_start(uint32 startadd,uint8 bpp) 
{
	LOG(1,"CRTC: setting card RAM to be displayed\n");

	/*figure out startadd value hardware needs*/
	/*switch(bpp)
	{
		case 8:case 24:
			startadd>>=1;
		case 16:
			startadd>>=1;
		case 32:
			startadd>>=1;
			break;
	}*/
	startadd>>=3;

	LOG(2,"CRTC: startadd: %x\n",startadd);
	LOG(2,"CRTC: frameRAM: %x\n",si->framebuffer);
	LOG(2,"CRTC: framebuffer: %x\n",si->fbc.frame_buffer);

	/*program the card!*/
	VGAW_I(CRTC,0xD,startadd&0xFF);
	VGAW_I(CRTC,0xC,(startadd&0xFF00)>>8);
	VGAW_I(CRTCEXT,8,((startadd&0x200000)>>21));
	VGAW_I(CRTCEXT,0,(VGAR_I(CRTCEXT,0)&0xB0)|((startadd&0xF0000)>>16)|((startadd&0x10000)>>14));
	return B_OK;
}

status_t g400_crtc_mem_priority(uint8 HIPRILVL)
{
	LOG(2,"CRTC: Setting memory priority level: %x\n",HIPRILVL);
	
	switch (HIPRILVL)
	{
	case 0:
		VGAW_I(CRTCEXT,6,0x00);
		break;
	case 1:case 2:case 3:
		VGAW_I(CRTCEXT,6,0x10|HIPRILVL);
		break;
	case 4:case 5:case 6:case 7:
		VGAW_I(CRTCEXT,6,0x20|HIPRILVL);
		break;
	default:
		LOG(3,"CRTC: Memory priority level violation: %x\n",HIPRILVL);
		return B_ERROR;
	}

	return B_OK;
}
