/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.

	modification for G400 driver - Mark Watson
*/

#include "acc_std.h"

/*
	Enable/Disable interrupts.  Just a wrapper around the
	ioctl() to the kernel driver.
*/
static void interrupt_enable(bool flag) {
	status_t result;
	g400_set_bool_state sbs;

	/* set the magic number so the driver knows we're for real */
	sbs.magic = G400_PRIVATE_DATA_MAGIC;
	sbs.do_it = flag;
	/* contact driver and get a pointer to the registers and shared data */
	result = ioctl(fd, G400_RUN_INTERRUPTS, &sbs, sizeof(sbs));
}
static void do_set_display_mode(display_mode *dm) {
	uint8 colour_depth;
	status_t result;
	uint32 hsync_pos;
	uint32 vsync_pos;
	double HIPRILVL;
	double Tmclk,Tpix,temp;	

	/* disable interrupts using the kernel driver */
	interrupt_enable(false);

	/* set the display mode*/
	hsync_pos = (dm->timing.flags&B_POSITIVE_HSYNC)!=0;
	vsync_pos = (dm->timing.flags&B_POSITIVE_VSYNC)!=0;
	result = g400_crtc_set_timing
	(
		dm->timing.h_display,
		dm->timing.h_sync_start,
		dm->timing.h_sync_end,
		dm->timing.h_total,
		dm->timing.v_display,
		dm->timing.v_sync_start,
		dm->timing.v_sync_end,
		dm->timing.v_total,
		hsync_pos,
		vsync_pos
	);

	/*set the pixel clock PLL*/
	if (g400_dac_set_pix_pll((dm->timing.pixel_clock)/1000.0)==B_ERROR)
	{
		LOG(3,"CRTC: error setting pixel clock\n");
	}

	/*work out the colour depth*/
	switch(dm->space)
	{
	case B_CMAP8:
		colour_depth=8;
		g400_dac_mode(BPP8,1.0);
		break;
	case B_RGB15_LITTLE:
		colour_depth=16;/*this is correct, because same memory footprint*/
		g400_dac_mode(BPP15,1.0);
		break;
	case B_RGB16_LITTLE:
		colour_depth=16;
		g400_dac_mode(BPP16,1.0);
		break;
	case B_RGB32_LITTLE:
		colour_depth=32;
		g400_dac_mode(BPP32,1.0);
		break;
	default:
		colour_depth=32;
	}

	/*calculate if high priority request are needed and how many*/
	Tpix = 1000000.0/(dm->timing.pixel_clock);
	Tmclk = si->ps.mem_clk_period;
	temp = (128/colour_depth);
	HIPRILVL = 64*Tmclk + (1-46*(temp))*Tpix;
	HIPRILVL/= -8*Tpix*temp;
	g400_crtc_mem_priority((uint8)HIPRILVL);

	/*Tell card what memory to display*/
	g400_crtc_set_display_start
	(
		(si->fbc.frame_buffer)-(si->framebuffer),
		colour_depth
	);

	g400_crtc_set_display_pitch
	(
		dm->virtual_width,
		colour_depth
	);

	/*update curent mode*/
	si->dm=*dm;
	si->fbc.bytes_per_row=dm->virtual_width*colour_depth/8;

	/*set up acceleration for this mode*/
	g400_acc_init();
	
	/* enable interrupts using the kernel driver */
	interrupt_enable(true);
}

/*
	The exported mode setting routine.  First validate the mode, then call our
	private routine to hammer the registers.
*/
status_t SET_DISPLAY_MODE(display_mode *mode_to_set) {
	display_mode bounds, target;

	/* ask for the specific mode */
	target = bounds = *mode_to_set;
	if (PROPOSE_DISPLAY_MODE(&target, &bounds, &bounds) == B_ERROR)
		return B_ERROR;
	do_set_display_mode(&target);
	return B_OK;
}

/*
	Set which pixel of the virtual frame buffer will show up in the
	top left corner of the display device.  Used for page-flipping
	games and virtual desktops.
*/
status_t MOVE_DISPLAY(uint16 h_display_start, uint16 v_display_start) {
	uint8 colour_depth;
	uint32 startadd;
	/*
	Many devices have limitations on the granularity of the horizontal offset.
	Make any checks for this here.  A future revision of the driver API will
	add a hook to return the granularity for a given display mode.
	*/
	/* G400 handles multiples of 8 for 8bit, 4 for 16bit, 2 for 32 bit */
	switch(si->dm.space)
	{
	case B_CMAP8:
		colour_depth=8;
		if (h_display_start&0x07) return B_ERROR;
		break;
	case B_RGB15_LITTLE: case B_RGB16_LITTLE:
		colour_depth=16;
		if (h_display_start&0x03) return B_ERROR;
		break;
	case B_RGB32_LITTLE:
		colour_depth=32;
		if (h_display_start&0x01) return B_ERROR;
		break;
	default:
		return B_ERROR;
	}

	/* do not run past end of display */
	if ((si->dm.timing.h_display + h_display_start) > si->dm.virtual_width)
		return B_ERROR;
	if ((si->dm.timing.v_display + v_display_start) > si->dm.virtual_height)
		return B_ERROR;

	/* everybody remember where we parked... */
	si->dm.h_display_start = h_display_start;
	si->dm.v_display_start = v_display_start;

	/* actually set the registers */
	startadd=v_display_start*(si->dm.virtual_width*colour_depth)>>3;
	startadd+=h_display_start;
	startadd+=(si->fbc.frame_buffer)-(si->framebuffer);
	interrupt_enable(false);
	g400_crtc_set_display_start(startadd,colour_depth);
	interrupt_enable(true);
	return B_OK;
}

/*
	Set the indexed color palette.
*/
void SET_INDEXED_COLORS(uint count, uint8 first, uint8 *color_data, uint32 flags) {
	int i;
	uint8 *r,*g,*b;
	/*
	Some cards use the indexed color regisers in the DAC for gamma correction.
	If this is true with your device (and it probably is), you need to protect
	against setting these registers when not in an indexed mode.
	*/
	if (si->dm.space != B_CMAP8) return;

	/*
	There isn't any need to keep a copy of the data being stored, as the app_server
	will set the colors each time it switches to an 8bpp mode, or takes ownership
	of an 8bpp mode after a GameKit app has used it.
	*/
	r=si->color_data;
	g=r+256;
	b=g+256;

	i=first;
	while (count--)
	{
		r[i]=*color_data++;
		g[i]=*color_data++;
		b[i]=*color_data++;
		i++;	
	}
	g400_dac_palette(r,g,b);
}


/* masks for DPMS control bits */
enum {
	H_SYNC_OFF = 0x01,
	V_SYNC_OFF = 0x02,
	DISPLAY_OFF = 0x04,
	BITSMASK = H_SYNC_OFF | V_SYNC_OFF | DISPLAY_OFF
};

/*
	Put the display into one of the Display Power Management modes.
*/
status_t SET_DPMS_MODE(uint32 dpms_flags) {
	interrupt_enable(false);

	/* now pick one of the DPMS configurations */
	switch(dpms_flags) {
		case B_DPMS_ON:	/* H: on, V: on */
			g400_crtc_dpms(1,1,1);
			break;
		case B_DPMS_STAND_BY:
			g400_crtc_dpms(0,0,1);
			break;
		case B_DPMS_SUSPEND:
			g400_crtc_dpms(0,1,0);
			break;
		case B_DPMS_OFF: /* H: off, V: off, display off */
			g400_crtc_dpms(0,0,0);
			break;
		default:
			LOG(3,"SetDisplayMode: Invalid DPMS settings\n");
			interrupt_enable(true);
			return B_ERROR;
	}
	interrupt_enable(true);
	return B_OK;
}

/*
	Report device DPMS capabilities.  G400 supports only D0,D3
*/
uint32 DPMS_CAPABILITIES(void) {
	return 	B_DPMS_ON | B_DPMS_STAND_BY | B_DPMS_SUSPEND|B_DPMS_OFF;
}


/*
	Return the current DPMS mode.
*/
uint32 DPMS_MODE(void) {
	uint8 display,h,v;
	
	interrupt_enable(false);
	g400_crtc_dpms_fetch(&display,&h,&v);
	interrupt_enable(true);

	if (display&h&v)
		return B_DPMS_ON;
	else if(v)
		return B_DPMS_STAND_BY;
	else if(h)
		return B_DPMS_SUSPEND;
	else
		return B_DPMS_OFF;
}
